// Excel Karşılaştırma Aracı - JavaScript (temiz yeniden yazım)

document.addEventListener('DOMContentLoaded', function() {
	// Form olayları (varsa)
    const uploadForm = document.getElementById('uploadForm');
    if (uploadForm) {
        uploadForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleFileUpload();
        });
    }
    
    // Dosya seçildiğinde validasyon
    const fileInputs = document.querySelectorAll('input[type="file"]');
    fileInputs.forEach(input => {
        input.addEventListener('change', function() {
            validateFile(this);
        });
    });
    
    // Mesajları otomatik gizle
    const messages = document.querySelectorAll('.message');
    messages.forEach(message => {
        setTimeout(() => {
            message.style.opacity = '0';
            message.style.transition = 'opacity 0.5s';
            setTimeout(() => {
                message.remove();
            }, 500);
        }, 5000);
    });

	// JSON'dan tabloyu yükle (otomatik yükleme kaldırıldı)
	// loadAndCompareData();
});

// Dosya validasyonu
function validateFile(input) {
    const file = input.files[0];
    const statusDiv = document.getElementById(input.id + '_status');
    
    if (!statusDiv) {
        console.error('Status div not found for input:', input.id);
        return false;
    }
    
    if (!file) {
        statusDiv.innerHTML = '';
        return false;
    }
    
    // Dosya boyutu kontrolü (10MB)
    const maxSize = 10 * 1024 * 1024;
    if (file.size > maxSize) {
        statusDiv.innerHTML = '<span class="error">Dosya boyutu çok büyük (Max: 10MB)</span>';
        return false;
    }
    
    // Dosya formatı kontrolü
    const allowedExtensions = ['xlsx', 'xls'];
    const extension = file.name.split('.').pop().toLowerCase();
    
    if (!allowedExtensions.includes(extension)) {
        statusDiv.innerHTML = '<span class="error">Sadece .xlsx ve .xls dosyaları kabul edilir</span>';
        return false;
    }
    
    // Başarılı
    statusDiv.innerHTML = '<span class="success">✓ ' + file.name + ' (' + formatFileSize(file.size) + ')</span>';
    return true;
}

// AJAX dosya yükleme
function handleFileUpload() {
    const form = document.getElementById('uploadForm');
    const formData = new FormData(form);
    const submitBtn = document.getElementById('submitBtn');
    const progressContainer = document.getElementById('progressContainer');
    const progressFill = document.getElementById('progressFill');
    const progressText = document.getElementById('progressText');
    
    // Validasyon
    const bankFile = document.getElementById('bank_file');
    const stockFile = document.getElementById('stock_file');
    
    if (!bankFile.files[0] || !stockFile.files[0]) {
        showMessage('Lütfen her iki dosyayı da seçin.', 'error');
        return;
    }
    
    if (!validateFile(bankFile) || !validateFile(stockFile)) {
        showMessage('Lütfen geçerli dosyalar seçin.', 'error');
        return;
    }
    
    // UI durumunu güncelle
    submitBtn.disabled = true;
    submitBtn.textContent = 'İşleniyor...';
    progressContainer.style.display = 'block';
    
    // Progress bar animasyonu
    let progress = 0;
    const progressInterval = setInterval(() => {
        progress += Math.random() * 15;
        if (progress > 90) progress = 90;
        progressFill.style.width = progress + '%';
    }, 200);
    
    // AJAX isteği
    fetch('php/api.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
        clearInterval(progressInterval);
        progressFill.style.width = '100%';
        
            if (data.success) {
            progressText.textContent = 'Tamamlandı!';
            showMessage(data.message, 'success');
            // JSON dosyalarını oku ve karşılaştır
            loadAndCompareData();
            } else {
            showMessage(data.error, 'error');
            resetForm();
            }
        })
        .catch(error => {
        clearInterval(progressInterval);
            console.error('Error:', error);
        showMessage('Sunucu hatası oluştu. Lütfen tekrar deneyin.', 'error');
        resetForm();
    });
}

// Form sıfırlama
function resetForm() {
    const submitBtn = document.getElementById('submitBtn');
    const progressContainer = document.getElementById('progressContainer');
    
    submitBtn.disabled = false;
    submitBtn.textContent = 'Dosyaları Karşılaştır';
    progressContainer.style.display = 'none';
}

// Mesaj gösterme
function showMessage(message, type) {
    // Eski mesajları kaldır
    const existingMessages = document.querySelectorAll('.message');
    existingMessages.forEach(msg => msg.remove());
    
    // Yeni mesaj oluştur
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${type}`;
    messageDiv.textContent = message;
    
    // Mesajı ekle
    const main = document.querySelector('main');
    main.insertBefore(messageDiv, main.firstChild);
    
    // Otomatik gizle
    setTimeout(() => {
        messageDiv.style.opacity = '0';
        messageDiv.style.transition = 'opacity 0.5s';
        setTimeout(() => {
            messageDiv.remove();
        }, 500);
    }, 5000);
}

// Dosya boyutu formatla
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// JSON dosyalarını yükle ve karşılaştır
async function loadAndCompareData() {
    try {
        const [bankResponse, stockResponse] = await Promise.all([
            fetch('assets/json/bank_data.json'),
            fetch('assets/json/stock_data.json')
        ]);
        
        const bankData = await bankResponse.json();
        const stockData = await stockResponse.json();
        
		const resultRows = buildBestMatches(bankData, stockData);
        
        showResults({
            bank_count: bankData.length,
            stock_count: stockData.length,
			matches_count: resultRows.filter(r => r.isMatch).length,
			matches: resultRows
        });
    } catch (error) {
        console.error('JSON yükleme hatası:', error);
        showMessage('Veri yükleme hatası. Lütfen tekrar deneyin.', 'error');
        resetForm();
    }
}

// En iyi eşleşmeleri çıkar ve tüm satırları kapsa
function buildBestMatches(bankData, stockData) {
	const round2 = v => Math.round(v * 100) / 100;

	// Tutar bazında gruplar
	const amountToBank = new Map();
	const amountToStock = new Map();
	bankData.forEach((b, i) => {
		const k = round2(b.amount);
		if (!amountToBank.has(k)) amountToBank.set(k, []);
		amountToBank.get(k).push({ ...b, __i: i });
	});
	stockData.forEach((s, j) => {
		const k = round2(s.amount);
		if (!amountToStock.has(k)) amountToStock.set(k, []);
		amountToStock.get(k).push({ ...s, __j: j });
	});

	// Aday kenarları oluştur (aynı tutar şart)
	const edges = [];
	amountToBank.forEach((banks, amount) => {
		const stocks = amountToStock.get(amount) || [];
		banks.forEach(b => {
			stocks.forEach(s => {
				const sim = calculateNameSimilaritySimple(s.account, b.account);
				edges.push({ amount, bi: b.__i, sj: s.__j, sim });
			});
		});
	});

	// Benzerliğe göre sırala ve greedy eşleştir
	edges.sort((a, b) => b.sim - a.sim);
	const matchedBank = new Set();
	const matchedStock = new Set();
	const pairs = [];
	for (const e of edges) {
		if (matchedBank.has(e.bi) || matchedStock.has(e.sj)) continue;
		matchedBank.add(e.bi);
		matchedStock.add(e.sj);
		pairs.push(e);
	}

	// Sonuç satırları: önce eşleşenler, sonra eşleşmeyen banka, sonra eşleşmeyen ML
	const result = [];
	pairs.forEach(p => {
		const b = bankData[p.bi];
		const s = stockData[p.sj];
		result.push({
			bank_row: b.row,
			stock_row: s.row,
			bank_account: b.account,
			stock_account: s.account,
			bank_amount: b.amount,
			stock_amount: s.amount,
			similarity: Math.round(p.sim * 100),
			bank_data: b.data,
			stock_data: s.data,
			bank_headers: b.headers,
			stock_headers: s.headers,
                isMatch: true
		});
	});

	// Eşleşmeyen banka
	bankData.forEach((b, i) => {
		if (!matchedBank.has(i)) {
			result.push({
				bank_row: b.row,
				stock_row: '-',
				bank_account: b.account,
				stock_account: '-',
				bank_amount: b.amount,
				stock_amount: 0,
				similarity: 0,
				bank_data: b.data,
				stock_data: null,
				bank_headers: b.headers,
				stock_headers: null,
				isMatch: false
            });
        }
    });
    
	// Eşleşmeyen ML
	stockData.forEach((s, j) => {
		if (!matchedStock.has(j)) {
			result.push({
				bank_row: '-',
				stock_row: s.row,
				bank_account: '-',
				stock_account: s.account,
				bank_amount: 0,
				stock_amount: s.amount,
				similarity: 0,
				bank_data: null,
				stock_data: s.data,
				bank_headers: null,
				stock_headers: s.headers,
				isMatch: false
			});
		}
	});

	// Eşleşenleri en üste al, benzerliğe göre sırala
	result.sort((a, b) => {
		if (a.isMatch !== b.isMatch) return a.isMatch ? -1 : 1;
		return b.similarity - a.similarity;
	});

	return result;
}

// Basitleştirilmiş isim benzerliği (temiz ve agresif olmayan)
function calculateNameSimilaritySimple(a, b) {
	const s1 = normalizeText(a || '');
	const s2 = normalizeText(b || '');
	if (!s1 || !s2) return 0;
	if (s1 === s2) return 1;
	return calculateWordSimilarity(s1, s2);
}

// Kelime bazlı benzerlik (ağırlıklı Levenshtein + alt string)

// Kelime bazlı benzerlik hesaplama
function calculateWordSimilarity(str1, str2) {
    const words1 = str1.split(' ').filter(w => w.length > 1);
    const words2 = str2.split(' ').filter(w => w.length > 1);
    
    if (words1.length === 0 || words2.length === 0) return 0;
    
    let totalMatches = 0;
    let totalWeight = 0;
    
    words1.forEach(word1 => {
        let bestMatch = 0;
        const wordWeight = getWordWeight(word1);
        
        words2.forEach(word2 => {
            const similarity = calculateWordMatch(word1, word2);
            if (similarity > bestMatch) {
                bestMatch = similarity;
            }
        });
        
        totalMatches += bestMatch * wordWeight;
        totalWeight += wordWeight;
    });
    
    return totalWeight > 0 ? totalMatches / totalWeight : 0;
}

// Kelime ağırlığı (uzun kelimeler daha önemli)
function getWordWeight(word) {
    const length = word.length;
    if (length <= 2) return 0.5;
    if (length <= 4) return 1.0;
    if (length <= 6) return 1.5;
    return 2.0;
}

// İki kelime arasındaki benzerlik
function calculateWordMatch(word1, word2) {
    if (word1 === word2) return 1.0;
    
    const len1 = word1.length;
    const len2 = word2.length;
    
    if (len1 === 0 || len2 === 0) return 0;
    
    // Çok kısa kelimeler için tam eşleşme gerekli
    if (len1 <= 2 || len2 <= 2) {
        return word1 === word2 ? 1.0 : 0;
    }
    
    // Levenshtein distance
    const maxLen = Math.max(len1, len2);
    const distance = levenshteinDistance(word1, word2);
    const levenshteinScore = 1 - (distance / maxLen);
    
    // Jaro-Winkler
    const jaroScore = jaroWinkler(word1, word2);
    
    // Alt string eşleşmesi
    let substringScore = 0;
    if (word1.includes(word2) || word2.includes(word1)) {
        substringScore = 0.8;
    }
    
    return Math.max(levenshteinScore, jaroScore, substringScore);
}

// Eski arabirimlerle uyumluluk için
function calculateSimilarity(str1, str2) { return calculateNameSimilaritySimple(str1, str2); }

// Türkçe karakter normalizasyonu
function normalizeText(text) {
    const turkishChars = {
        'ç': 'c', 'ğ': 'g', 'ı': 'i', 'ö': 'o', 'ş': 's', 'ü': 'u',
        'Ç': 'C', 'Ğ': 'G', 'İ': 'I', 'Ö': 'O', 'Ş': 'S', 'Ü': 'U'
    };
    
    return text.toLowerCase()
        .replace(/[çğıöşüÇĞİÖŞÜ]/g, char => turkishChars[char] || char)
        .replace(/[^\w\s]/g, '')
        .replace(/\s+/g, ' ')
        .trim();
}

// Levenshtein distance
function levenshteinDistance(str1, str2) {
    const matrix = [];
    
    for (let i = 0; i <= str2.length; i++) {
        matrix[i] = [i];
    }
    
    for (let j = 0; j <= str1.length; j++) {
        matrix[0][j] = j;
    }
    
    for (let i = 1; i <= str2.length; i++) {
        for (let j = 1; j <= str1.length; j++) {
            if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
                matrix[i][j] = matrix[i - 1][j - 1];
            } else {
                matrix[i][j] = Math.min(
                    matrix[i - 1][j - 1] + 1,
                    matrix[i][j - 1] + 1,
                    matrix[i - 1][j] + 1
                );
            }
        }
    }
    
    return matrix[str2.length][str1.length];
}

// Jaro-Winkler benzerliği
function jaroWinkler(s1, s2) {
    if (s1 === s2) return 1.0;
    
    const len1 = s1.length;
    const len2 = s2.length;
    
    if (len1 === 0 || len2 === 0) return 0.0;
    
    const matchWindow = Math.floor(Math.max(len1, len2) / 2) - 1;
    const s1Matches = new Array(len1).fill(false);
    const s2Matches = new Array(len2).fill(false);
    
    let matches = 0;
    let transpositions = 0;
    
    // Eşleşen karakterleri bul
    for (let i = 0; i < len1; i++) {
        const start = Math.max(0, i - matchWindow);
        const end = Math.min(i + matchWindow + 1, len2);
        
        for (let j = start; j < end; j++) {
            if (s2Matches[j] || s1[i] !== s2[j]) continue;
            s1Matches[i] = true;
            s2Matches[j] = true;
            matches++;
            break;
        }
    }
    
    if (matches === 0) return 0.0;
    
    // Transpozisyonları say
    let k = 0;
    for (let i = 0; i < len1; i++) {
        if (!s1Matches[i]) continue;
        while (!s2Matches[k]) k++;
        if (s1[i] !== s2[k]) transpositions++;
        k++;
    }
    
    const jaro = (matches / len1 + matches / len2 + (matches - transpositions / 2) / matches) / 3;
    
    // Winkler bonusu
    let prefix = 0;
    for (let i = 0; i < Math.min(4, Math.min(len1, len2)); i++) {
        if (s1[i] === s2[i]) prefix++;
        else break;
    }
    
    return jaro + (0.1 * prefix * (1 - jaro));
}

function showResults(data) {
    // Eski sonuçları kaldır
    const existingResults = document.querySelector('.results-section');
    if (existingResults) {
        existingResults.remove();
    }
    
    // Sonuç bölümünü oluştur
    const resultsSection = document.createElement('div');
    resultsSection.className = 'results-section';
    
    resultsSection.innerHTML = `
        <h2>Karşılaştırma Sonuçları</h2>
        <div class="result-actions">
            <button onclick="exportToExcel()" class="btn btn-secondary">Excel'e Aktar</button>
            <button onclick="printResults()" class="btn btn-secondary">Yazdır</button>
        </div>
        <div class="results-summary">
            <div class="summary-item">
                <span class="label">Banka Dosyası Kayıt Sayısı:</span>
                <span class="value">${data.bank_count}</span>
            </div>
            <div class="summary-item">
                <span class="label">ML Dosyası Kayıt Sayısı:</span>
                <span class="value">${data.stock_count}</span>
            </div>
        </div>
        <div id="results-table">
			${generateResultsTable(data.matches)}
        </div>
    `;
    
    // Ana bölüme ekle
    const main = document.querySelector('main');
    main.appendChild(resultsSection);
    
    // Progress bar'ı gizle
    resetForm();
}

function generateResultsTable(matches) {
    const matchedCount = matches.filter(m => m.isMatch).length;
	const total = matches.length;
    
    let html = '<div class="results-table">';
    html += '<div class="results-summary">';
	html += `<div class="summary-item"><span class="label">Eşleşen:</span> <span class="value success">${matchedCount}</span></div>`;
	html += `<div class="summary-item"><span class="label">Toplam Satır:</span> <span class="value">${total}</span></div>`;
    html += '</div>';
    
    html += '<div class="table-container">';
    html += '<table class="table">';
	html += '<thead><tr><th>#</th><th>Banka Satır</th><th>Banka Açıklama</th><th>Banka Tutar</th><th>ML Satır</th><th>ML Açıklama</th><th>ML Tutar</th><th>Eşleşme Oranı</th><th>Detay</th></tr></thead>';
    html += '<tbody>';
    
	matches.forEach((row, index) => {
		const i = index + 1;
		const similarityClass = getSimilarityClass(row.similarity || 0);
		const bankAmount = row.bank_amount || 0;
		const stockAmount = row.stock_amount || 0;
		const bankAmountClass = bankAmount < 0 ? 'negative-amount' : bankAmount > 0 ? 'positive-amount' : 'zero-amount';
		const stockAmountClass = stockAmount < 0 ? 'negative-amount' : stockAmount > 0 ? 'positive-amount' : 'zero-amount';
		
        html += `<tr class="match-row ${similarityClass}">
			<td class="row-number">${i}</td>
			<td class="bank-row bank-section">${row.bank_row ?? '-'}</td>
			<td class="bank-account bank-section">${row.bank_account ?? '-'}</td>
			<td class="amount bank-section ${bankAmountClass}">${formatAmount(bankAmount)}</td>
			<td class="stock-row stock-section">${row.stock_row ?? '-'}</td>
			<td class="stock-account stock-section">${row.stock_account ?? '-'}</td>
			<td class="amount stock-section ${stockAmountClass}">${formatAmount(stockAmount)}</td>
			<td class="similarity"><span class="similarity-badge similarity-${similarityClass}">%${row.similarity || 0}</span></td>
			<td class="actions"><button onclick="toggleDetails(${index})" class="btn btn-sm btn-detail">Detay</button></td>
        </tr>`;
        
        html += `<tr id="detail-${index}" class="detail-row" style="display: none;">
			<td colspan="9">
                <div class="detail-content">
                    <div class="detail-header">
						<h4>Detay</h4>
						<button onclick="toggleDetails(${index})" class="btn btn-sm btn-close">Kapat</button>
                    </div>
                    <div class="detail-body">
                        <div class="detail-section">
							<h5>Banka</h5>
                            <div class="data-preview">
								<strong>Satır:</strong> ${row.bank_row ?? '-'}<br>
								<strong>Hesap:</strong> ${row.bank_account ?? '-'}<br>
								<strong>Tutar:</strong> ${formatAmount(row.bank_amount)}
                            </div>
							${row.bank_data ? `<details><summary>Ham Veri</summary><pre>${escapeHtml(JSON.stringify(row.bank_data, null, 2))}</pre></details>` : ''}
                        </div>
                        <div class="detail-section">
							<h5>ML</h5>
							<div class="data-preview">
								<strong>Satır:</strong> ${row.stock_row ?? '-'}<br>
								<strong>Hesap:</strong> ${row.stock_account ?? '-'}<br>
								<strong>Tutar:</strong> ${formatAmount(row.stock_amount)}
                            </div>
							${row.stock_data ? `<details><summary>Ham Veri</summary><pre>${escapeHtml(JSON.stringify(row.stock_data, null, 2))}</pre></details>` : ''}
                        </div>
                    </div>
                </div>
            </td>
        </tr>`;
    });
    
    html += '</tbody></table></div></div>';
    return html;
}

// Benzerlik sınıfını belirle
function getSimilarityClass(similarity) {
    if (similarity >= 90) return 'excellent';
    if (similarity >= 80) return 'very-good';
    if (similarity >= 70) return 'good';
    if (similarity >= 60) return 'fair';
    return 'poor';
}

// Detay satırını göster/gizle
function toggleDetails(index) {
    const detailRow = document.getElementById(`detail-${index}`);
    if (detailRow.style.display === 'none') {
        detailRow.style.display = 'table-row';
    } else {
        detailRow.style.display = 'none';
    }
}

// Excel'e aktarma fonksiyonu
function exportToExcel() {
    const table = document.querySelector('.results-table');
    if (!table) {
        alert('Aktarılacak veri bulunamadı.');
        return;
    }
    
    // Tabloyu CSV formatına çevir
    let csv = '';
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cells = row.querySelectorAll('th, td');
        const rowData = Array.from(cells).map(cell => {
            let text = cell.textContent.trim();
            // Virgül ve tırnak işaretlerini escape et
            text = text.replace(/"/g, '""');
            if (text.includes(',') || text.includes('"') || text.includes('\n')) {
                text = '"' + text + '"';
            }
            return text;
        });
        csv += rowData.join(',') + '\n';
    });
    
    // CSV dosyasını indir
    const blob = new Blob(['\ufeff' + csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', 'excel_karsilastirma_sonuclari.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Yazdırma fonksiyonu
function printResults() {
    const resultsSection = document.querySelector('.results-section');
    if (!resultsSection) {
        alert('Yazdırılacak veri bulunamadı.');
        return;
    }
    
    // Yazdırma için yeni pencere aç
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
        <html>
        <head>
            <title>Excel Karşılaştırma Sonuçları</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                th { background-color: #f2f2f2; }
                .amount { text-align: right; font-weight: bold; }
                .account-name { font-size: 12px; }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <h1>Excel Karşılaştırma Sonuçları</h1>
            <p>Yazdırma Tarihi: ${new Date().toLocaleString('tr-TR')}</p>
            ${resultsSection.innerHTML}
        </body>
        </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    printWindow.print();
}

// Yardımcılar
function formatAmount(v) {
	if (typeof v !== 'number' || isNaN(v)) return '-';
	
	// Türk Lirası formatı: 1.234.567,89 ₺
	const formatted = new Intl.NumberFormat('tr-TR', {
		minimumFractionDigits: 2,
		maximumFractionDigits: 2
	}).format(v);
	
	return formatted + ' ₺';
}

function escapeHtml(str) {
	return (str || '')
		.replace(/&/g, '&amp;')
		.replace(/</g, '&lt;')
		.replace(/>/g, '&gt;');
}